/******************************************************************************
 * Název projektu: Aplet pro demonstraci Boyerova-Mooreova algoritmu
 * Balíček: boyermooredemo
 * Soubor: BarevnyTextPane.java
 * Datum:  13.4.2008
 * Poslední změna: 18.4.2008
 * Autor:  Jaroslav Dytrych xdytry00
 *
 * Popis: Třída pro vytvoření textového panelu, který umožňuje vkládat bloky 
 *        textu, u nichž lze nastavit barvu písma a pozadí a zda je písmo tučné.
 *             Bloky jsou číslovány dle pořadí vložení do panelu a po jejich
 *        vložení je možné měnit barvu písma a pozadí bloku s daným číslem.
 *
 ******************************************************************************/

/**
 * @file BarevnyTextPane.java
 * 
 * @brief Třída BarevnyTextPane - textový panel s barevnými bloky textu
 */

package boyermooredemo;

import java.awt.Color;
import java.util.TreeMap;
import javax.swing.JOptionPane;
import javax.swing.JTextPane;
import javax.swing.text.BadLocationException;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;

/**
 * Třída pro vytvoření textového panelu, který umožňuje vkládat bloky textu, 
 * u nichž lze nastavit barvu písma a pozadí a zda je písmo tučné.
 *      Bloky jsou číslovány dle pořadí vložení do panelu a po jejich vložení 
 * je možné měnit barvu písma a pozadí bloku s daným číslem.
 * 
 * @brief Textový panel s barevnými bloky textu
 */
public class BarevnyTextPane extends JTextPane {
  /**
   * Třída pro uchování informací o úseku textu
   * 
   * @brief Třída pro uchování informací o úseku textu
   */
  class usekTextu
  {
    /** Začátek úseku textu */
    public int zacatek;
    /** Délka úseku textu */
    public int delka;
    /** Barva, se kterou byl úsek textu vložen do panelu */
    public Color vychoziBarva;
    /** Barva pozadí, se kterou byl úsek textu vložen do panelu */
    public Color vychoziBarvaPozadi;
    /** Určení, zda je úsek textu vypsán tučným písmem */
    public boolean tucne;

    /**
     * Konstruktor třídy, uloží informace o úseku
     *
     * @param zacatekUseku Pozice začátku úseku v dokumentu
     * @param delkaUseku Délka úseku
     * @param barva Barva úseku
     * @param barvaPozadi Barva pozadí úseku
     * @param tucneP Určuje, zda je písmo tučné
     */
    usekTextu(int zacatekUseku, int delkaUseku, Color barva, Color barvaPozadi, 
              boolean tucneP)
    {
      zacatek = zacatekUseku;
      delka = delkaUseku;
      vychoziBarva = barva;
      vychoziBarvaPozadi = barvaPozadi;
      tucne = tucneP;
    }
  }  // class usekTextu
  
  private StyledDocument dokument;  // dokument v panelu
  private int pocetUseku = 0;  // počet úseků textu vypsaných v panelu
  private TreeMap<Integer,usekTextu> useky;
  
  /**
   * Konstruktor třídy, vytvoří textový panel a provede inicializaci
   */
  public BarevnyTextPane() 
  {
    super();
    this.dokument = (StyledDocument)this.getDocument();  // získá dokument
    this.useky = new TreeMap<Integer,usekTextu>();  // vytvoří kontejner pro úseky textu
    this.pocetUseku = 0;  // zatím není uložen žádný úsek textu
  }
  
  /**
   * Metoda pro vyprázdnění panelu
   */
  public void vyprazdniPanel()
  {
    try {
      this.dokument.remove(0,dokument.getLength());  // vyprázdní dokument
    } catch (BadLocationException e) {
       JOptionPane.showMessageDialog(this, "Došlo k chybě při úpravě pole s algoritmem.",
                                     "Chyba",JOptionPane.ERROR_MESSAGE);
    }
    this.useky = new TreeMap<Integer,usekTextu>();  // vyprázdní kontejner s informacemi o úsecích
    this.pocetUseku = 0;  // počet uložených úseků je nyní 0
  }
  
  /**
   * Metoda pro přidání textu do panelu
   * 
   * @param vkladanyText Vkládaný text
   * @param barva Barva textu
   * @param barvaPozadi Barva pozadí
   * @param tucnePismo Určuje, zda je písmo tučné
   */
  public void pridejText(String vkladanyText, Color barva, Color barvaPozadi, 
                         boolean tucnePismo)
  {
    // vytvoří stylový objekt
    Style styl = this.dokument.addStyle("Styl" + this.pocetUseku, null);
    // Nastaví neproporcionální písmo
    StyleConstants.setFontFamily(styl, "SansSerif");
    // Nastaví velikost písma
    StyleConstants.setFontSize(styl, 12);
    // Nastaví barvu písma
    StyleConstants.setForeground(styl, barva);  
    // Nastaví barvu pozadí
    StyleConstants.setBackground(styl, barvaPozadi);
    if (tucnePismo)
    {  // pokud má být písmo tučné, nastaví jej
      StyleConstants.setBold(styl, true);
    }
    
    // uloží informace o úseku do kontejneru
    this.useky.put(this.pocetUseku,new usekTextu(dokument.getLength(),
                   vkladanyText.length(),barva,barvaPozadi,tucnePismo));
    this.pocetUseku++;  // byl přidán další úsek
       
    // Vloží úsek textu do dokumentu
    try {
      dokument.insertString(dokument.getLength(), vkladanyText, styl);
    } catch (BadLocationException e) {
      JOptionPane.showMessageDialog(this, "Došlo k chybě při vypisování algoritmu.",
                                    "Chyba",JOptionPane.ERROR_MESSAGE);
    }
  }  // public void pridejText()
  
  /**
   * Metoda pro nastavení barvy úseku textu
   *
   * @param cisloUseku Číslo úseku textu
   * @param barva Nová barva
   * @param barvaPozadi Nová barva pozadí
   */
  public void nastavBarvu(int cisloUseku, Color barva, Color barvaPozadi)
  {
    // zjistí informace o úseku
    usekTextu informace = useky.get(cisloUseku);
    
    dokument.removeStyle("Styl" + cisloUseku);  // odebere starý styl úseku

    // vytvoří stylový objekt
    Style styl = this.dokument.addStyle("Styl" + cisloUseku, null);
    // Nastaví neproporcionální písmo
    StyleConstants.setFontFamily(styl, "SansSerif");
    // Nastaví velikost písma
    StyleConstants.setFontSize(styl, 12);
    // Nastaví barvu písma
    StyleConstants.setForeground(styl, barva);  
    // Nastaví barvu pozadí
    StyleConstants.setBackground(styl, barvaPozadi);
    if (informace.tucne)
    {  // pokud má být písmo tučné, nastaví jej
      StyleConstants.setBold(styl, true);
    }
    
    // zjistí text úseku, odebere jej z dokumentu a vloží v nové barvě
    String textUseku = "";
    try {
      textUseku = dokument.getText(informace.zacatek,informace.delka);  // zjistí text
      dokument.remove(informace.zacatek,informace.delka);  // odebere
      dokument.insertString(informace.zacatek, textUseku, styl);  // vloží přebarvený
    } catch (BadLocationException e) {
       JOptionPane.showMessageDialog(this, "Došlo k chybě při vypisování algoritmu.",
                                     "Chyba",JOptionPane.ERROR_MESSAGE);
    }
  }  // public void nastavBarvu
  
   /**
    * Metoda pro nastavení výchozí barvy úseku textu
    * (nastaví úseku tu barvu, se kterou byl vložen do panelu)
    *
    * @param cisloUseku Číslo úseku textu
    */
   public void nastavVychoziBarvu(int cisloUseku)
   {
     // zjistí informace o úseku textu
     usekTextu informace = useky.get(cisloUseku);
     
     dokument.removeStyle("Styl" + cisloUseku);  // odebere starý styl úseku

     // vytvoří stylový objekt
     Style styl = this.dokument.addStyle("Styl" + cisloUseku, null);
     // Nastaví neproporcionální písmo
     StyleConstants.setFontFamily(styl, "SansSerif");
     // Nastaví velikost písma
     StyleConstants.setFontSize(styl, 12);
     // Nastaví barvu písma
     StyleConstants.setForeground(styl, informace.vychoziBarva);  
     // Nastaví barvu pozadí
     StyleConstants.setBackground(styl, informace.vychoziBarvaPozadi);
     if (informace.tucne)
     {  // pokud má být písmo tučné, nastaví jej
       StyleConstants.setBold(styl, true);
     }
     
     // zjistí text úseku, odebere jej z dokumentu a vloží v nové barvě
     String textUseku = "";
     try {
       textUseku = dokument.getText(informace.zacatek,informace.delka);  // zjistí text
       dokument.remove(informace.zacatek,informace.delka);  // odebere
       dokument.insertString(informace.zacatek, textUseku, styl);  // vloží přebarvený
     } catch (BadLocationException e) {
        JOptionPane.showMessageDialog(this, "Došlo k chybě při vypisování algoritmu.",
                                      "Chyba",JOptionPane.ERROR_MESSAGE);
     }
   }  // public void nastavBarvu
  
}  // public class barevnyTextPane

/*** Konec souboru BarevnyTextPane.java ***/
