/******************************************************************************
 * Název projektu: Aplet pro demonstraci Boyerova-Mooreova algoritmu
 * Balíček: boyermooredemo
 * Soubor: Tabulka.java
 * Datum:  11.4.2008
 * Poslední změna: 18.4.2008
 * Autor:  Jaroslav Dytrych xdytry00
 *
 * Popis: Třída pro vytvoření tabulky, která má záhlaví v jednotlivých řádcích 
 *        prvního sloupce, nebo vůbec. 
 *             V případě, že se text nevejde do buňky tabulky, buňky jsou 
 *        automaticky rozšířeny. První a poslední sloupec mohou mít jinou šířku,
 *        než sloupce ve zbytku tabulky.
 *             Rozměry tabulky lze za běhu měnit. Tabulka by měla být umístěna 
 *        ve skrolovacím panelu, jehož skrolovatelnou oblast a krok skrolování 
 *        nastavuje v rámci implementace rozhraní Scrollable.
 *
 ******************************************************************************/

/**
 * @file Tabulka.java
 * 
 * @brief Třída Tabulka - vytvoření tabulky se záhlavími řádků
 */

package boyermooredemo;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;

import javax.swing.JPanel;
import javax.swing.Scrollable;

/**
 * Třída pro vytvoření tabulky, která má záhlaví v jednotlivých řádcích prvního 
 * sloupce, nebo vůbec. 
 *      V případě, že se text nevejde do buňky tabulky, buňky jsou automaticky 
 * rozšířeny. První a poslední sloupec mohou mít jinou šířku, než sloupce 
 * ve zbytku tabulky.
 *      Rozměry tabulky lze za běhu měnit. Tabulka by měla být umístěna 
 * ve skrolovacím panelu, jehož skrolovatelnou oblast a krok skrolování 
 * nastavuje v rámci implementace rozhraní Scrollable.
 * 
 * @brief Třída pro vytvoření tabulky se záhlavími řádků
 */
public class Tabulka extends JPanel implements Scrollable {
  /**
   * Proměnná pro počet řádků tabulky
   */
  private int pocetRadku = 0;
  /**
   * Proměnná pro počet sloupců tabulky
   */
  private int pocetSloupcu = 0;
  /**
   * Tloušťka ohraničení buněk tabulky, 
   * kolem celé tabulky bude ohraničení dvojnásobné.
   */
  private int ohraniceni = 0;
  /**
   * pole s obsahy buněk tabulky
   */
  private String[][] bunky;
  /**
   * Pole s barvami buněk
   * 
   * 3. rozměr rozlišuje barvu písma [0] a barvu pozadí [1]
   */
  private Color[][][] barvyBunek;
  /**
   * Barva pozadí tabulky
   */
  private Color barvaPozadi;
  /**
   * Barva písma v tabulce
   */
  private Color barvaPisma;
  /**
   * Šířka plátna, na kterém bude tabulka vykreslena
   */
  private int sirkaPlatna = 0;
  /**
   * Výška plátna, na kterém bude tabulka vykreslena
   */
  private int vyskaPlatna = 0;
  /**
   * Okraje buňky
   */
  private int okrajeBunky = 4;
  /**
   * Šířka prázdného řetězce v px
   */
  private int sirkaPrazdneho = 5;
  /**
   * Šířka buňky tabulky
   */
  private int sirkaBunky = 5;
  /**
   * Výška buňky tabulky
   */
  private int vyskaBunky = 10;
  /**
   * Metrika fontu - obsahuje metody pro zjišťování šířky a výšky řetězce
   */
  private FontMetrics metrikaFontu = null;
  /**
   * Proměnná, která určuje, zda bude 1. sloupec hlavička tabulky, 
   * tedy zda mít 1. sloupec jinou šířku
   */
  private boolean prvniHlavicka = false;
  /**
   * Šířka 1. sloupce tabulky
   */
  private int sirkaPrvniho = 5;
  /**
   * Proměnná, která určuje, zda bude mít poslední sloupec tabulky jinou šířku
   */
  private boolean posledniJiny = false;
  /**
   * Šířka posledního sloupce tabulky
   */
  private int sirkaPosledniho = 5;
    
  /**
   * Konstanta pro směr posunu vlevo
   */
  public static final int POSUN_VLEVO = 0;
  /**
   * Konstanta pro směr posunu vpravo
   */
  public static final int POSUN_VPRAVO = 1;
    
  /**
   * Konstruktor třídy, vytvoří tabulku a provede inicializaci
   *
   * @param radky Počet řádků tabulky
   * @param sloupce Počet sloupců tabulky
   * @param ohraniceniTabuky Tloušťka ohraničení buněk tabulky a současně 
   *                          1/2 tloušťky ohraničení celé tabulky
   * @param okrajeBunkyTab Okraje buňky v tabulce
   * @param barvaPozadiTab Barva pozadí tabulky
   * @param barvaPismaTab Barva písma v tabulce
   * @param prvniHlavickaTab Určuje, zda bude 1. sloupec sloužit jako hlavička tabulky
   * @param posledniSlJiny Určuje, zda bude mít poslední sloupec tabulky jinou šířku
   */
  public Tabulka (int radky, int sloupce, int ohraniceniTabuky, int okrajeBunkyTab, 
                  Color barvaPozadiTab, Color barvaPismaTab, 
                  boolean prvniHlavickaTab, boolean posledniSlJiny) 
  {
    this.setDoubleBuffered(true);  // zapne double-buffer
    this.ohraniceni = ohraniceniTabuky;  // nastaví ohraničení tabulky
    this.barvaPozadi = barvaPozadiTab;  // nastaví nbarvu pozadí
    this.barvaPisma = barvaPismaTab;  // nastaví nbarvu písma
    this.okrajeBunky = okrajeBunkyTab;  // nastaví okraje buňky
    this.prvniHlavicka = prvniHlavickaTab;  // nastaví chování 1. sloupce
    this.posledniJiny = posledniSlJiny;  // nastaví chování posledního sloupce
    // nastaví rozměry tabulky
    this.pocetRadku = radky;
    this.pocetSloupcu = sloupce;
    // zjistí metriku fontu
    this.setFont(new Font("Arial",Font.PLAIN,12));
    this.metrikaFontu = this.getFontMetrics(this.getFont());
    if (prvniHlavickaTab)
    {  // pokud první sloupec bude tvořit hlavičku tabulky
      // upraví šířku 1. sloupce (přidá šířku 2. ohraničovací čáry)
      this.sirkaPrvniho += 2 * this.ohraniceni;
    }
    
    // nastaví výchozí rozměry plátna  
    this.sirkaPlatna = 10;
    this.vyskaPlatna = 10;
      
    this.vyskaBunky = this.metrikaFontu.getMaxAscent();  // určí výšku buňky

    this.bunky = new String[radky][sloupce];  // inicializace pole obsahů buněk
    this.barvyBunek = new Color[radky][sloupce][2];  // inicializace pole barev buněk
    
    for (int i = 0; i < radky; i++)
    {  // procházení tabulky po řádcích
      for (int j = 0; j < sloupce; j++)
      {  // procházení tabulky po sloupcích
        this.bunky[i][j] = "";  // nastaví obsah buňky
        this.barvyBunek[i][j][0] = barvaPismaTab;  // nastaví barvu písma
        this.barvyBunek[i][j][1] = barvaPozadiTab;  // nastaví barvu pozadí
      }
    }
       
    // vypočítá rozměry plátna
    // (vnější rámeček má dvojnásobný okraj, kolem tabulky je okraj 5 px)
    this.sirkaPlatna = sloupce * (this.sirkaBunky + ohraniceniTabuky + this.okrajeBunky * 2) 
                       + 4 * ohraniceniTabuky + 10;
    this.vyskaPlatna = radky * (this.vyskaBunky + ohraniceniTabuky + this.okrajeBunky * 2) 
                       + 4 * ohraniceniTabuky + 10;
  }  // public Tabulka 
    
  /**
   * Metoda pro vyprázdnění tabulky a nastavení výchozí barvy všech buněk
   * 
   * @param vcetnePrvniho Pokud je false, první sloupec se ponechá beze změny
   */
  public void vyprazdniTabulku(boolean vcetnePrvniho)
  {
    int zacatek;
    if (vcetnePrvniho)
    {  // pokud se má vymazat vše
      zacatek = 0;
      if (this.prvniHlavicka)
      {  // pokud je 1. sloupec hlavička
        this.sirkaPrvniho = this.sirkaPrazdneho + 2 * this.ohraniceni;  // šířka 1. buňky
      }
      else
      {  // pokud poslední sloupec není hlavička
        this.sirkaPrvniho = this.sirkaPrazdneho;  // šířka 1. buňky
      }
    }  // pokud se má vymazat vše
    else
    {  // pokud se má 1. sloupec ponechat
      zacatek = 1;
    }
    this.sirkaBunky = this.sirkaPrazdneho;  // nastaví výchozí šířku buněk
    this.sirkaPosledniho = this.sirkaPrazdneho;  // nastaví šířku posledního sloupce
    
    for (int i = 0; i < this.pocetRadku; i++)
    {  // procházení tabulky po řádcích
      for (int j = zacatek; j < this.pocetSloupcu; j++)
      {  // procházení tabulky po sloupcích
        this.bunky[i][j] = "";  // nastaví obsah buňky
        this.barvyBunek[i][j][0] = this.barvaPisma;  // nastaví barvu písma
        this.barvyBunek[i][j][1] = this.barvaPozadi;  // nastaví barvu pozadí
      }
    }
    // přepočítá rozměry plátna
    // (vnější rámeček má dvojnásobný okraj, kolem tabulky je okraj 5 px)
    this.sirkaPlatna = (this.pocetSloupcu - 2) * (this.sirkaBunky + this.ohraniceni + this.okrajeBunky * 2) 
      + this.sirkaPrvniho + this.sirkaPosledniho + 2 * this.ohraniceni + this.okrajeBunky * 4 
      + 4 * this.ohraniceni + 10;
    this.vyskaPlatna = this.pocetRadku * (this.vyskaBunky + this.ohraniceni + this.okrajeBunky * 2) 
      + 4 * this.ohraniceni + 10;
    this.prekresliTabulku();  // překreslí tabulku
  }  // public void vyprazdniTabulku()
    
  /**
   * Metoda pro nastavení obsahu buňky
   *
   * @param radek Řádek, na kterém se buňka nachází
   * @param sloupec Sloupec, na kterém se buňka nachází
   * @param hodnota Nová hodnota v buňce
   */
  public void nastavObsah(int radek, int sloupec, String hodnota) 
  {
    int sirkaObsahu;
    int porovnavanaSirka;
    if (this.prvniHlavicka && sloupec == 0)
    {  // pokud se bude aktualizovat šířka 1. sloupce
      porovnavanaSirka = this.sirkaPrvniho;
    }
    else if (this.posledniJiny && sloupec == (this.pocetSloupcu - 1))
    {  // pokud se bude aktualizovat šířka posledního sloupce
      porovnavanaSirka = this.sirkaPosledniho;
    }
    else
    {  // pokud se bude aktualizovat šířka buňky
      porovnavanaSirka = this.sirkaBunky;
    }
    this.bunky[radek][sloupec] = hodnota;  // nastaví obsah buňky
    if (hodnota.length() > 0)
    {  // pokud bnka není prázdná
      sirkaObsahu = metrikaFontu.stringWidth(hodnota);  // zjistí šířku řetězce
    }
    else
    {  // pokud je buňka prázdná
      sirkaObsahu = this.sirkaPrazdneho;
    }
    if (sirkaObsahu > porovnavanaSirka)
    {  // pokud je třeba rozšířit buňky
      if (this.prvniHlavicka && sloupec == 0)
      {  // pokud se jedná o buňku hlavičky
        this.sirkaPrvniho = sirkaObsahu;  // hozšíří hlavičku
        // upraví šířku 1. sloupce (přidá šířku 2. ohraničovací čáry)
        this.sirkaPrvniho += 2 * this.ohraniceni;
      }
      else if (this.posledniJiny && sloupec == (this.pocetSloupcu - 1))
      {  // pokud se jedná o buňku posledního sloupce
        this.sirkaPosledniho = sirkaObsahu;  // hozšíří buňky
      }
      else
      {  // pokud se jedná o obecnou buňku
        this.sirkaBunky = sirkaObsahu;  // hozšíří buňky
        if (!this.prvniHlavicka)
        {  // pokud 1. sloupec obsahuje obecné buňky
          this.sirkaPrvniho = sirkaObsahu;
        }
        if (!this.posledniJiny)
        {  // pokud poslední sloupec obsahuje obecné buňky
          this.sirkaPosledniho = sirkaObsahu;
        }
      }  // pokud se jedná o obecnou buňku
      // přepočítá rozměry plátna
      // (vnější rámeček má dvojnásobný okraj, kolem tabulky je okraj 5 px)
      this.sirkaPlatna = (this.pocetSloupcu - 2) * (this.sirkaBunky + this.ohraniceni + this.okrajeBunky * 2) 
        + this.sirkaPrvniho + this.sirkaPosledniho + 2 * this.ohraniceni + this.okrajeBunky * 4 
        + 4 * this.ohraniceni + 10;
      this.vyskaPlatna = this.pocetRadku * (this.vyskaBunky + this.ohraniceni + this.okrajeBunky * 2) 
        + 4 * this.ohraniceni + 10;
    }
    this.prekresliTabulku();  // překreslí tabulku
  }  // public void nastavObsah()
    
  /**
   * Metoda pro nastavení barev buňky
   *
   * @param radek Řádek, na kterém se buňka nachází
   * @param sloupec Sloupec, na kterém se buňka nachází
   * @param barvaPozadiB Nová barva pozadí buňky
   * @param barvaPismaB Nová barva písma v buňce
   */
  public void nastavBarvu(int radek, int sloupec, Color barvaPozadiB, Color barvaPismaB)
  {
    this.barvyBunek[radek][sloupec][0] = barvaPismaB;  // nastaví barvu buňky
    this.barvyBunek[radek][sloupec][1] = barvaPozadiB;  // nastaví barvu písma
    this.prekresliTabulku();  // překreslí tabulku
  }
   
   
  /**
   * Metoda pro minimalizaci šířky buňky tabulky
   */
  public void minimalizaceSirky()
  {
    int minSirka = this.sirkaPrazdneho;
    int sirkaObsahu;
    int zacatek;  // pomocná proměnná
    if (this.prvniHlavicka)
    {  // pokud je 1. sloupec hlavička
      zacatek = 1;
      for (int i = 0; i < this.pocetRadku; i++)
      {  // nalezení nejširší buňky sloupce hlavičky
        sirkaObsahu = this.metrikaFontu.stringWidth(bunky[i][0]);
        if (sirkaObsahu > minSirka)
        {
          minSirka = sirkaObsahu;
        }
      }
      this.sirkaPrvniho = minSirka;  // nastaví novou šířku hlavičky
      // upraví šířku 1. sloupce (přidá šířku 2. ohraničovací čáry)
      this.sirkaPrvniho += 2 * this.ohraniceni;
    }
    else
    {  // pokud 1. sloupec není hlavička
      zacatek = 0;
    }
    int konec = this.pocetSloupcu;
    if (this.posledniJiny)
    {  // pokud má poslední sloupec jinou šířku
      konec--;  // poslední sloupec se nebude procházet
      for (int i = 0; i < this.pocetRadku; i++)
      {  // nalezení nejširší buňky posledního sloupce
        sirkaObsahu = this.metrikaFontu.stringWidth(bunky[i][konec]);
        if (sirkaObsahu > minSirka)
        {
          minSirka = sirkaObsahu;
        }
      }
      this.sirkaPosledniho = minSirka;  // nastaví novou šířku posledního sloupce
    }
    // nalezení nejširšího obsahu buňky
    for (int i = 0; i < this.pocetRadku; i++)
    {  // procházení po řádcích
      for (int j = zacatek; j < konec; j++)
      {
        sirkaObsahu = this.metrikaFontu.stringWidth(bunky[i][j]);
        if (sirkaObsahu > minSirka)
        {
          minSirka = sirkaObsahu;
        }
      }
    }  // procházení po řádcích
    this.sirkaBunky = minSirka;  // nastaví novou šířku buňky
    if (!this.prvniHlavicka)
    {  // pokud 1. sloupec netvoří hlavičku
      this.sirkaPrvniho = minSirka;
    }
    if (!this.posledniJiny)
    {  // pokud poslední sloupec není jiný
      this.sirkaPosledniho = minSirka;
    }
    // přepočítá rozměry plátna
    // (vnější rámeček má dvojnásobný okraj, kolem tabulky je okraj 5 px)
    this.sirkaPlatna = (this.pocetSloupcu - 2) * (this.sirkaBunky + this.ohraniceni + this.okrajeBunky * 2) 
      + this.sirkaPrvniho + this.sirkaPosledniho + 2 * this.ohraniceni + this.okrajeBunky * 4 
      + 4 * this.ohraniceni + 10;
    this.vyskaPlatna = this.pocetRadku * (this.vyskaBunky + this.ohraniceni + this.okrajeBunky * 2) 
      + 4 * this.ohraniceni + 10;
    this.prekresliTabulku();  // překreslí tabulku
  }  // public void minimalizaceSirky
   
   
  /**
   * Metoda pro změnu rozměrů tabulky
   *
   * @param novyPocetRadku
   * @param novyPocetSloupcu
   */
  public void zmenRozmery (int novyPocetRadku, int novyPocetSloupcu)
  {
    String[][] noveBunky;  // pomocná pole
    Color[][][] noveBarvyBunek;
    
    if (novyPocetRadku > this.pocetRadku)
    {  // pokud má být počet řádků větší, je nutné upravit pole
      // inicializace nového pole obsahů buněk
      noveBunky = new String[novyPocetRadku][this.pocetSloupcu];
      // inicializace nového pole barev buněk
      noveBarvyBunek = new Color[novyPocetRadku][this.pocetSloupcu][2];
      // uložení obsahu a barev buněk do nového pole
      for (int i = 0; i < this.pocetRadku; i++)
      {  // procházení tabulky po řádcích
        for (int j = 0; j < this.pocetSloupcu; j++)
        {  // procházení tabulky po sloupcích
          noveBunky[i][j] = this.bunky[i][j];  // nastaví obsah buňky
          noveBarvyBunek[i][j][0] = this.barvyBunek[i][j][0];  // nastaví barvu písma
          noveBarvyBunek[i][j][1] = this.barvyBunek[i][j][1];  // nastaví barvu pozadí
        }
      }
      // nastaví výchozí stav nových buněk
      for (int i = this.pocetRadku; i < novyPocetRadku; i++)
      {  // procházení tabulky po řádcích
        for (int j = 0; j < this.pocetSloupcu; j++)
        {  // procházení tabulky po sloupcích
          this.bunky[i][j] = "";  // nastaví obsah buňky
          this.barvyBunek[i][j][0] = this.barvaPisma;  // nastaví barvu písma
          this.barvyBunek[i][j][1] = this.barvaPozadi;  // nastaví barvu pozadí
        }
      }
      this.pocetRadku = novyPocetRadku;  // aktualizije počet řádků
      // záměna polí
      this.bunky = noveBunky;
      this.barvyBunek = noveBarvyBunek;
    }  // pokud má být počet řádků větší, je nutné upravit pole
    else if (novyPocetRadku < this.pocetRadku)
    {  // pokud má být počet menší, stačí jej pouze nastavit
      this.pocetRadku = novyPocetRadku;
    }
    
    if (novyPocetSloupcu > this.pocetSloupcu)
    {  // pokud má být počet sloupců větší, je nutné upravit pole
      // inicializace nového pole obsahů buněk
      noveBunky = new String[this.pocetRadku][novyPocetSloupcu];
      // inicializace nového pole barev buněk
      noveBarvyBunek = new Color[this.pocetRadku][novyPocetSloupcu][2];
      // uložení obsahu a barev buněk do nového pole
      for (int i = 0; i < this.pocetRadku; i++)
      {  // procházení tabulky po řádcích
        for (int j = 0; j < this.pocetSloupcu; j++)
        {  // procházení tabulky po sloupcích
          noveBunky[i][j] = this.bunky[i][j];  // nastaví obsah buňky
          noveBarvyBunek[i][j][0] = this.barvyBunek[i][j][0];  // nastaví barvu písma
          noveBarvyBunek[i][j][1] = this.barvyBunek[i][j][1];  // nastaví barvu pozadí
        }
      }
      // nastaví výchozí stav nových buněk
      for (int i = 0; i < this.pocetRadku; i++)
      {  // procházení tabulky po řádcích
        for (int j = this.pocetSloupcu; j < novyPocetSloupcu; j++)
        {  // procházení tabulky po sloupcích
          noveBunky[i][j] = new String("");  // nastaví obsah buňky
          noveBarvyBunek[i][j][0] = this.barvaPisma;  // nastaví barvu písma
          noveBarvyBunek[i][j][1] = this.barvaPozadi;  // nastaví barvu pozadí
        }
      }
      this.pocetSloupcu = novyPocetSloupcu;  // aktualizije počet řádků
      // záměna polí
      this.bunky = noveBunky;
      this.barvyBunek = noveBarvyBunek;
    }  // pokud má být počet sloupců větší, je nutné upravit pole
    else if (novyPocetSloupcu < this.pocetSloupcu)
    {  // pokud má být počet menší, stačí jej pouze nastavit
      this.pocetSloupcu = novyPocetSloupcu;
    }
    
    // přepočítá rozměry plátna
    // (vnější rámeček má dvojnásobný okraj, kolem tabulky je okraj 5 px)
    this.sirkaPlatna = (this.pocetSloupcu - 2) * (this.sirkaBunky + this.ohraniceni + this.okrajeBunky * 2) 
      + this.sirkaPrvniho + this.sirkaPosledniho + 2 * this.ohraniceni + this.okrajeBunky * 4 
      + 4 * this.ohraniceni + 10;
    this.vyskaPlatna = this.pocetRadku * (this.vyskaBunky + this.ohraniceni + this.okrajeBunky * 2) 
      + 4 * this.ohraniceni + 10;
    this.prekresliTabulku();  // překreslí tabulku
  }  // private void zmenRozmery()
   
  /**
   * Metoda pro posun obsahu buněk v řádku tabulky
   * - pouze pro tabulky bez zvláštního prvního a posledního sloupce
   * - zleva (zprava) přidá prázdná políčka, zprava (zleva) přebytečná políčka zahodí
   * - neovlivňuje barvy buněk, posouvá pouze obsah
   * 
   * @param radek Řádek tabulky
   * @param smer Směr posuvu: POSUN_VLEVO - vlevo, POSUN_VPRAVO - vpravo
   * @param pocet Počet posuvů
   */
  public void posunRadek(int radek, int smer, int pocet)
  {
    if (smer == this.POSUN_VLEVO)
    {  // pokud se má posouvat vlevo
      for (int i = 0; i < (this.pocetSloupcu - pocet); i++)
      {  // posun
          this.bunky[radek][i] = this.bunky[radek][i+pocet];
      }
      for (int i = (this.pocetSloupcu - pocet); i < this.pocetSloupcu; i++)
      {  // doplnění prázdných buněk
          this.bunky[radek][i] = "";
      }
    }
    else
    {  // pokud se má posouvat vpravo
      for (int i = (this.pocetSloupcu - 1); (i - pocet) >= 0; i--)
      {  // posun
          this.bunky[radek][i] = this.bunky[radek][i-pocet];
      }
      for (int i = 0; i < pocet; i++)
      {  // doplnění prázdných buněk
          this.bunky[radek][i] = "";
      }
    }
  }  // public void posunRadek()
   
  /**
   * Metoda pro překreslení tabulky
   */
  private void prekresliTabulku()
  {
    // nastavení rozměrů skrolované oblasti
    this.setMinimumSize(new Dimension(this.sirkaPlatna,this.vyskaPlatna));
    this.setPreferredSize(new Dimension(this.sirkaPlatna,this.vyskaPlatna));
    this.setSize(this.sirkaPlatna,this.vyskaPlatna);
    this.revalidate();  // vyhodnocení rozměrů
    this.repaint();  // překreslení
  }
  
  /**
   * Metoda pro vykreslení tabulky
   * 
   * @param g Grafický objekt
   */
  public void paintComponent(Graphics g)
  {           
    int zacatek;  // pomocná proměnná
    if (this.prvniHlavicka)
    {  // pokud je 1. sloupec hlavička
      zacatek = 1;
    }
    else
    {  // pokud 1. sloupec není hlavička
      zacatek = 0;
    }
    int konec;  // pomocná proměnná
    if (this.posledniJiny)
    {  // pokud je poslední sloupec jiný
      konec = this.pocetSloupcu - 1;
    }
    else
    {  // pokud poslední sloupec není jiný
      konec = this.pocetSloupcu;
    }
    super.paintComponent(g);
    Graphics2D g2 = (Graphics2D)g;  // přetypování na vylepšenou grafiku
    int invariant1 = 5 + 2 * this.ohraniceni - this.okrajeBunky;  // invariant cyklu
    int invariant1P = 5 + 2 * this.ohraniceni - this.okrajeBunky 
      + zacatek * (this.sirkaPrvniho - this.sirkaBunky);  // invariant cyklu
    // 2. invariant cyklu (šířka/výška okrajů v buňce)
    int invariant2 = 2 * this.okrajeBunky;
    // 3. invariant cyklu (šířka buňky bez okrajů včetně ohraničení)
    int invariant3 = this.sirkaBunky + this.ohraniceni;
    // 4. invariant cyklu (výška buňky bez okrajů včetně ohraničení)
    int invariant4 = this.vyskaBunky + this.ohraniceni;
    // 5. invariant cyklu (šířka buňky včetně okrajů a ohraničení)
    int invariant5 = this.sirkaBunky + this.ohraniceni + invariant2;
    // 6. invariant cyklu (výška buňky včetně okrajů a ohraničení)
    int invariant6 = this.vyskaBunky + this.ohraniceni + invariant2;
    for (int i = 0; i < this.pocetRadku; i++)
    {  // vykreslení jednotlivých řádků tabulky
      for (int j = zacatek; j < konec; j++)
      {  // vykreslení jednotlivých buněk v řádku tabulky
        // vykreslí pozadí buňky
        g2.setColor(this.barvyBunek[i][j][1]);
        g2.fillRect(invariant1P + this.okrajeBunky + j * invariant5,
                    invariant1 + this.okrajeBunky + i * invariant6,
                    this.sirkaBunky + invariant2,
                    this.vyskaBunky + invariant2);
        // vykreslí obsah buňky
        g2.setColor(this.barvyBunek[i][j][0]);
        g2.drawString(this.bunky[i][j],
                      invariant1P + (j + 1) * invariant2 + j * invariant3,
                      invariant1 + (i + 1) * invariant2 + i * invariant4 + this.vyskaBunky);
      }
    }  // vykreslení jednotlivých řádků tabulky
    if (this.prvniHlavicka)
    {  // pokud je 1. sloupec hlavička, vykreslí jej
      for (int i = 0; i < this.pocetRadku; i++) 
      {
        // vykreslí pozadí buňky
        g2.setColor(this.barvyBunek[i][0][1]);
        g2.fillRect(invariant1 + this.okrajeBunky,
                    invariant1 + this.okrajeBunky + i * invariant6,
                    this.sirkaPrvniho + invariant2,
                    this.vyskaBunky + invariant2);
        // vykreslí obsah buňky
        g2.setColor(this.barvyBunek[i][0][0]);
        g2.drawString(this.bunky[i][0],
                      invariant1 + invariant2,
                      invariant1 + (i + 1) * invariant2 + i * invariant4 + this.vyskaBunky);
        
      }
    }  // pokud je 1. sloupec hlavička, vykreslí jej
    if (this.posledniJiny)
    {  // pokud je poslední sloupec jiný, vykreslí jej
      for (int i = 0; i < this.pocetRadku; i++) 
      {  // procházení řádků
        // vykreslí pozadí buňky
        g2.setColor(this.barvyBunek[i][konec][1]);
        g2.fillRect(invariant1P + this.okrajeBunky + konec * invariant5,
                    invariant1 + this.okrajeBunky + i * invariant6,
                    this.sirkaPosledniho + invariant2,
                    this.vyskaBunky + invariant2);            
        // vykreslí obsah buňky
        g2.setColor(this.barvyBunek[i][konec][0]);
        g2.drawString(this.bunky[i][konec],
                      invariant1P + (konec + 1) * invariant2 + konec * invariant3,
                      invariant1 + (i + 1) * invariant2 + i * invariant4 + this.vyskaBunky);
      }  // procházení řádků
    }  // pokud je poslední sloupec jiný, vykreslí jej
      
    if (this.ohraniceni > 0)
    {  // pokud má tabulka ohraničení
      g2.setColor(new Color(0,0,0));  // ohraničení tabulky bude černé
      g2.setStroke(new BasicStroke(this.ohraniceni * 2));  // dvojnásobná tloušťka čáry
      // vykreslí ohraničení celé tabulky
      g2.drawRect(5 + this.ohraniceni, 
                  5 + this.ohraniceni, 
                  this.sirkaPlatna - 10 - 2 * this.ohraniceni, 
                  this.vyskaPlatna - 10 - 2 * this.ohraniceni);
      g2.setStroke(new BasicStroke(this.ohraniceni));  // nastaví tloušťku čáry

      invariant1 = 5 + 2 * this.ohraniceni;  // invariant cyklu
      invariant1P = 5 + 2 * this.ohraniceni + zacatek * (this.sirkaPrvniho - this.sirkaBunky);  // invariant cyklu
      if (this.ohraniceni > 1)
      {  // pokud je třeba posun na střed čáry
        invariant1 -= this.ohraniceni / 2;
        invariant1P -= this.ohraniceni / 2;
      }
      // 2. invariant cyklu (mezera mezi čarami)
      invariant2 = 2 * this.okrajeBunky + this.vyskaBunky + this.ohraniceni;
      // 3. invariant cyklu (x-ová souřadnice konce čáry)
      invariant3 = this.sirkaPlatna - 5 - 2 * this.ohraniceni;
      for (int i = 1; i < this.pocetRadku; i++)
      {  // vykreslí čáry mezi řádky
        g2.drawLine(invariant1, 
                    invariant1 + i * invariant2, 
                    invariant3,
                    invariant1 + i * invariant2);        
      }
      // 2. invariant cyklu (mezera mezi čarami)
      invariant2 = 2 * this.okrajeBunky + this.sirkaBunky + this.ohraniceni;
      // 3. invariant cyklu (y-ová souřadnice konce čáry)
      invariant3 = this.vyskaPlatna - 5 - 2 * this.ohraniceni;
      for (int i = 1 + zacatek; i < this.pocetSloupcu; i++)
      {  // vykreslí čáry mezi sloupci
        g2.drawLine(invariant1P + i * invariant2, 
                    invariant1, 
                    invariant1P + i * invariant2,
                    invariant3);
      }
      if (this.prvniHlavicka)
      {  // pokud je 1. sloupec hlavička, vykreslí jeho oddělení od zbytku tabulky
        g2.drawLine(invariant1 + 1 * 2 * this.okrajeBunky + this.sirkaPrvniho + this.ohraniceni, 
                    invariant1, 
                    invariant1 + 1 * 2 * this.okrajeBunky + this.sirkaPrvniho + this.ohraniceni,
                    invariant3);
        g2.drawLine(invariant1 + 1 * 2 * this.okrajeBunky + this.sirkaPrvniho - this.ohraniceni, 
                    invariant1, 
                    invariant1 + 1 * 2 * this.okrajeBunky + this.sirkaPrvniho - this.ohraniceni,
                    invariant3);            
      }
    }  // pokud má tabulka ohraničení
  }  // public void paintComponent()
  
  /**
   * Metoda pro zjištění aktuálního obsahu buňky
   *
   * @param radek Řádek, na kterém se buňka nachází
   * @param sloupec Sloupec, na kterém se buňka nachází
   * @return Vrací obsah požadované buňky
   */
  public String vratObsah(int radek, int sloupec)
  {
    return this.bunky[radek][sloupec];
  }
  
  /**
   * Metoda pro zjištění aktuální barvy buňky
   *
   * @param radek Řádek, na kterém se buňka nachází
   * @param sloupec Sloupec, na kterém se buňka nachází
   * @param pismoPozadi 0 - barva písma v buňce
   *                    1 - barva pozadí buňky
   * @return Vrací požadovanou barvu
   */
  public Color vratBarvu(int radek, int sloupec,int pismoPozadi)
  {
    return this.barvyBunek[radek][sloupec][pismoPozadi];
  }

  /**
   * Metoda pro zjištění počtu řádků tabulky
   * 
   * @return Vrací počet řádků tabulky
   */
  public int vratPocetRadkuTabulky()
  {
    return this.pocetRadku;
  }
  
  /**
   * Metoda pro zjištění počtu sloupců tabulky
   * 
   * @return Vrací počet sloupců tabulky
   */
  public int vratPocetSloupcuTabulky()
  {
    return this.pocetSloupcu;
  }

  /**
   * Metoda pro zjištění zobrazované oblasti ve skrolovacím panelu
   *
   * @return Vrací rozměry oblasti s tabulkou
   */
  public Dimension getPreferredScrollableViewportSize() {
    return new Dimension(this.sirkaPlatna,this.vyskaPlatna);
  }

  /**
   * Metoda pro nastavení jednotky skrolování
   *
   * @param visibleRect Viditelný obdélník
   * @param orientation Orientace
   * @param direction Směr
   * @return Vrací jednotku skrolování
   */
  public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation,
                                        int direction) {
    return this.sirkaBunky + this.ohraniceni + 2 * this.okrajeBunky;
  }

  /**
   * Metoda pro nastavení bloku skrolování
   *
   * @param visibleRect Viditelný obdélník
   * @param orientation Orientace
   * @param direction Směr
   * @return Vrací jednotku skrolování
   */
  public int getScrollableBlockIncrement(Rectangle visibleRect,
                                         int orientation, int direction) {
    return 5 * (this.sirkaBunky + this.ohraniceni + 2 * this.okrajeBunky);
  }

  /**
   * Metoda pro získání šířky pohledu skrolovací stopy
   *
   * @return Vrací šířku pohledu skrolovací stopy
   */
  public boolean getScrollableTracksViewportWidth() {
    return false;
  }
  
  /**
   * Metoda pro získání výšky pohledu skrolovací stopy
   *
   * @return Vrací výšku pohledu skrolovací stopy
   */
  public boolean getScrollableTracksViewportHeight() {
    return false;
  }
} // public class Tabulka

/*** Konec souboru Tabulka.java ***/
