/*
 * Project: Example game for creation of installer
 * Authors: Ing. Jan Kouřil
 *          Ing. Jaroslav Dytrych idytrych@fit.vutbr.cz
 * File: kufr.cpp
 */

/**
 * @file kufr.cpp
 *
 * @brief Main window of program
 */


#include "kufr.h"
#include "QMenuBar"
#include "QMenu"
#include "QAction"
#include "QKeySequence"
#include "QFileDialog"
#include "canvasWidget.h"
#include "QMessageBox"
#include "QDir"
#include "QApplication"
#include "QStatusBar"
#include "QLabel"

/**
 * Constructor of main window of program
 */
Kufr::Kufr(QWidget *parent)
    : QMainWindow(parent)
{
    // menu bar
    menubar=new QMenuBar(this);
    // File menu
    QMenu *file=new QMenu("File",menubar);
    QAction *addFile=new QAction("Add file",file);
    addFile->setShortcut(QKeySequence("Ctrl+F"));
    file->addAction(addFile);
    QAction *addDirectory=new QAction("Add directory",file);
    addDirectory->setShortcut(QKeySequence("Ctrl+D"));
    file->addAction(addDirectory);
    file->addSeparator();
    QAction *quit=new QAction("Quit",file);
    quit->setShortcut(QKeySequence("Ctrl+Q"));
    file->addAction(quit);
    menubar->addMenu(file);
    // Game menu
    QMenu *game=new QMenu("Game",menubar);
    QAction *start=new QAction("Start",game);
    start->setShortcut(QKeySequence("Ctrl+S"));
    game->addAction(start);
    game->addSeparator();
    QAction *next=new QAction("Next",game);
    next->setShortcut(QKeySequence(Qt::Key_Right));
    game->addAction(next);
    QAction *previous=new QAction("Previous",game);
    previous->setShortcut(QKeySequence(Qt::Key_Left));
    game->addAction(previous);
    game->addSeparator();
    QAction *clear=new QAction("Clear",game);
    clear->setShortcut(QKeySequence(Qt::Key_Return));
    game->addAction(clear);
    QAction *cover=new QAction("Cover",game);
    cover->setShortcut(QKeySequence(Qt::Key_Escape));
    game->addAction(cover);
    menubar->addMenu(game);
    // Grid menu
    QMenu *grid=new QMenu("Grid",menubar);
    four=new QAction("4x4",grid);
    four->setShortcut(QKeySequence(Qt::Key_4));
    four->setCheckable(true);
    grid->addAction(four);
    six=new QAction("6x6",grid);
    six->setShortcut(QKeySequence(Qt::Key_6));
    six->setCheckable(true);
    grid->addAction(six);
    eight=new QAction("8x8",grid);
    eight->setShortcut(QKeySequence(Qt::Key_8));
    eight->setCheckable(true);
    eight->setChecked(true);
    grid->addAction(eight);
    menubar->addMenu(grid);
    // Attach menu bar to this window
    this->setMenuBar(menubar);
    // Create canvas for picture
    canvas=new CanvasWidget(this);
    this->setCentralWidget(canvas);
    // Create status bar
    QStatusBar *status=new QStatusBar(this);
    this->setStatusBar(status);
    // Set window title and initial size
    setWindowTitle("Kufr");
    resize(800,600);
    // Set initial status
    QLabel *statusLabel=new QLabel("Ready.",this);
    status->addWidget(statusLabel,0);

    // Connect signals
    connect(addFile,SIGNAL(triggered()),this,SLOT(aFile()));
    connect(addDirectory,SIGNAL(triggered()),this,SLOT(aDirectory()));
    connect(quit,SIGNAL(triggered()),qApp,SLOT(quit()));
    
    connect(start,SIGNAL(triggered()),canvas,SLOT(start()));
    connect(next,SIGNAL(triggered()),canvas,SLOT(next()));
    connect(previous,SIGNAL(triggered()),canvas,SLOT(prev()));
    connect(clear,SIGNAL(triggered()),canvas,SLOT(clearMatrices()));
    connect(cover,SIGNAL(triggered()),canvas,SLOT(coverMatrices()));
    
    connect(four,SIGNAL(triggered()),this,SLOT(set4()));
    connect(six,SIGNAL(triggered()),this,SLOT(set6()));
    connect(eight,SIGNAL(triggered()),this,SLOT(set8()));
    
    connect(canvas,SIGNAL(resizeRequest(int,int)),this,SLOT(resizeCanvas(int,int)));
    connect(canvas,SIGNAL(statusRequest(QString)),statusLabel,SLOT(setText(QString)));
}  // Kufr::Kufr()

/**
 * Handler for Add file menu action
 */
void Kufr::aFile(){
    QString str=QFileDialog::getOpenFileName(NULL,"Open file","","Images (*.gif *.jpeg *.pbm *.bmp *.pgm *.ppm *.tiff *.xbm *.png *.xpm *.jpg)");
    if (!str.isEmpty()){
        canvas->addImage(str);
    }
}

/**
 * Handler for Add directory menu action
 */
void Kufr::aDirectory(){
    QString str=QFileDialog::getExistingDirectory(NULL,"Open directory");
    if (!str.isEmpty()){
        QDir dir(str);
        QStringList entries=dir.entryList(QStringList()<<"*.gif"<<"*.jpeg"<<"*.pbm"<<"*.bmp"<<"*.pgm"<<"*.ppm"<<"*.tiff"<<"*.xbm"<<"*.png"<<"*.xpm"<<"*.jpg");
        for (int i=0;i<entries.size();i++)
            canvas->addImage(str+"/"+entries[i]);
    }
}

/**
 * Handler for Grid 4x4 menu action
 */
void Kufr::set4(){
    four->setChecked(true);
    six->setChecked(false);
    eight->setChecked(false);
    canvas->setMode(4);
}

/**
 * Handler for Grid 6x6 menu action
 */
void Kufr::set6(){
    four->setChecked(false);
    six->setChecked(true);
    eight->setChecked(false);
    canvas->setMode(6);
}

/**
 * Handler for Grid 8x8 menu action
 */
void Kufr::set8(){
    four->setChecked(false);
    six->setChecked(false);
    eight->setChecked(false);
    canvas->setMode(8);
}

/**
 * Handler for canvas resize event
 * 
 * @param w Width
 * @param h height
 */
void Kufr::resizeCanvas(int w,int h){
    int m=eight->isChecked()?8:six->isChecked()?6:4;
    setFixedSize(this->width()+w/m*m-canvas->width(),this->height()+h/m*m-canvas->height());
}


