/*
 * Project: Example game for creation of installer
 * Authors: Ing. Jan Kouřil
 *          Ing. Jaroslav Dytrych idytrych@fit.vutbr.cz
 * File: kufr.java
 */

/**
 * @file kufr.java
 *
 * @brief Main window of program
 */

package cz.vutbr.fit.kufr;
 
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.FileFilter;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFileChooser;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.SystemColor;
import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * Main window of program
 *
 * @brief Main window of program
 */
public class kufr extends JFrame implements ActionListener{
    /** Menu items for matrice dimensions */
    private JCheckBoxMenuItem four, six, eight;
    /** Canvas for image */
    private canvasWidget canvas;
    /** File chooser */
    private JFileChooser chooser;
    /** Status bar */
    private StatusBar statusBar;
    
    /**
     * Constructor
     */
    public kufr(){
        super();
        // Create widgets
        chooser=new JFileChooser();
        canvas=new canvasWidget();
        canvas.setActionListener(this);
        JMenuBar menubar=new JMenuBar();
        // File menu
        JMenu file=new JMenu("File");
        menubar.add(file);
        JMenuItem addFile=new JMenuItem("Add file");
        addFile.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F,ActionEvent.CTRL_MASK));
        file.add(addFile);
        addFile.addActionListener(this);
        JMenuItem addDirectory=new JMenuItem("Add directory");
        addDirectory.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D,ActionEvent.CTRL_MASK));
        addDirectory.addActionListener(this);
        file.add(addDirectory);
        file.addSeparator();
        JMenuItem quit=new JMenuItem("Quit");
        quit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q,ActionEvent.CTRL_MASK));
        quit.addActionListener(this);
        file.add(quit);
        // Game menu
        JMenu game=new JMenu("Game");
        menubar.add(game);
        JMenuItem start=new JMenuItem("Start");
        start.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,ActionEvent.CTRL_MASK));
        start.addActionListener(canvas);
        game.add(start);
        game.addSeparator();
        JMenuItem next=new JMenuItem("Next");
        next.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT,0));
        next.addActionListener(canvas);
        game.add(next);
        JMenuItem prev=new JMenuItem("Prev");
        prev.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_LEFT,0));
        prev.addActionListener(canvas);
        game.add(prev);
        game.addSeparator();
        JMenuItem clear=new JMenuItem("Clear");
        clear.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,0));
        clear.addActionListener(canvas);
        game.add(clear);
        JMenuItem cover=new JMenuItem("Cover");
        cover.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE,0));
        cover.addActionListener(canvas);
        game.add(cover);
        // Grid menu
        JMenu grid=new JMenu("Grid");
        menubar.add(grid);
        four=new JCheckBoxMenuItem("4x4");
        four.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_NUMPAD4,0));
        four.addActionListener(this);
        grid.add(four);
        six=new JCheckBoxMenuItem("6x6");
        six.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_NUMPAD6,0));
        six.addActionListener(this);
        grid.add(six);
        eight=new JCheckBoxMenuItem("8x8");
        eight.setState(true);
        eight.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_NUMPAD8,0));
        eight.addActionListener(this);
        grid.add(eight); 
        // Attach menu bar to this window
        this.setJMenuBar(menubar);
        // Attach canvas for picture
        Container pane=this.getContentPane();
        pane.setLayout(new BorderLayout());
        pane.add(canvas);
        // Create status bar
        statusBar=new StatusBar();
        pane.add(statusBar, BorderLayout.SOUTH);
        // Set initial size of window
        this.resize(800, 600);
        // Set close action
        this.setDefaultCloseOperation(EXIT_ON_CLOSE);
    }

    /**
     * Action handler
     * 
     * @param ae Action event
     */
    @Override
    public void actionPerformed(ActionEvent ae) {
        String command=ae.getActionCommand();
        if ("Quit".equals(command)){
            dispose();
            System.exit(0);
        }
        else if ("Add directory".equals(command)){
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            if (chooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION){
                File f=chooser.getSelectedFile();
                File[] files=f.listFiles(new ImageFileFilter());
                for (int i=0;i<files.length;i++){
                    canvas.addImage(files[i]);
                }
            }
        }
        else if ("Add file".equals(command)){
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            if (chooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION){
                File f=chooser.getSelectedFile();
                canvas.addImage(f);
            }
        }
        else if ("4x4".equals(command)){
            set4();
        }
        else if ("6x6".equals(command)){
            set6();
        }
        else if ("8x8".equals(command)){
            set8();
        }
        else if (command.startsWith("Image")){
            statusBar.setText(command);
        }
        else if (command.startsWith("Resize")){
            String dimensions=command.substring(7);
            int pos=dimensions.indexOf(' ');
            int width=Integer.parseInt(dimensions.substring(0,pos));
            int height=Integer.parseInt(dimensions.substring(pos+1));
            resizeCanvas(width,height);
        }
    }
    
  /**
   * Resize canvas
   *
   * @param w Width
   * @param h height
   */
    private void resizeCanvas(int w, int h){
        int m=eight.getState()?8:six.getState()?6:4;
        this.setSize(getWidth()+w/m*m-canvas.getWidth(),getHeight()+h/m*m-canvas.getHeight());
    }

    /**
     * Sets matrice to 4x4
     */
    private void set4() {
        four.setState(true);
        six.setState(false);
        eight.setState(false);
        canvas.setMode(4);
    }

    /**
     * Sets matrice to 6x6
     */
    private void set6() {
        four.setState(false);
        six.setState(true);
        eight.setState(false);
        canvas.setMode(6);
    }

    /**
     * Sets matrice to 8x8
     */
    private void set8() {
        four.setState(false);
        six.setState(false);
        eight.setState(true);
        canvas.setMode(8);
    }
} // public class kufr

/**
 * Filter for image files
 */
class ImageFileFilter implements FileFilter {
  /**
   * Allowed extensions
   */
  private final String[] okFileExtensions = new String[]{"jpg", "png", "gif", "bmp"};

  /** 
   * Filtering method 
   */
  @Override
  public boolean accept(File file) {
    for (String extension : okFileExtensions) {
      if (file.getName().toLowerCase().endsWith(extension)) {
        return true;
      }
    }
    return false;
  }
}

/**
 * Status bar
 */
class StatusBar extends JPanel {
  /** Label */
  JLabel label;
  
  /** 
   * Constructor 
   */
  public StatusBar() {
    setLayout(new BorderLayout());
    setPreferredSize(new Dimension(10, 23));
    JPanel rightPanel = new JPanel(new BorderLayout());
    JLabel MyLabel=new JLabel(new AngledLinesWindowsCornerIcon());
    rightPanel.add(MyLabel, BorderLayout.SOUTH);
    rightPanel.setOpaque(false);
    add(rightPanel, BorderLayout.EAST);
    JPanel leftPanel = new JPanel(new BorderLayout());
    label=new JLabel("Ready.");
    leftPanel.add(label, BorderLayout.CENTER);
    leftPanel.setOpaque(false);
    add(leftPanel,BorderLayout.WEST);
    setBackground(SystemColor.control);
  }
  
  /**
   * Sets text
   * 
   * @param s New text
   */
  public void setText(String s){
      label.setText(s);
  }

  /**
   * Paints statusbar
   */
  @Override
  protected void paintComponent(Graphics g) {
    super.paintComponent(g);

    int y = 0;
    g.setColor(new Color(156, 154, 140));
    g.drawLine(0, y, getWidth(), y);
    y++;
    g.setColor(new Color(196, 194, 183));
    g.drawLine(0, y, getWidth(), y);
    y++;
    g.setColor(new Color(218, 215, 201));
    g.drawLine(0, y, getWidth(), y);
    y++;
    g.setColor(new Color(233, 231, 217));
    g.drawLine(0, y, getWidth(), y);

    y = getHeight() - 3;
    g.setColor(new Color(233, 232, 218));
    g.drawLine(0, y, getWidth(), y);
    y++;
    g.setColor(new Color(233, 231, 216));
    g.drawLine(0, y, getWidth(), y);
    y = getHeight() - 1;
    g.setColor(new Color(221, 221, 220));
    g.drawLine(0, y, getWidth(), y);

  }
}  // class StatusBar extends
  
/**
 * Corner icon
 */
class AngledLinesWindowsCornerIcon implements Icon {
  /** White color */
  private static final Color WHITE_LINE_COLOR = new Color(255, 255, 255);
  /** Gray color */
  private static final Color GRAY_LINE_COLOR = new Color(172, 168, 153);
  /** Width */
  private static final int WIDTH = 13;
  /** Height */
  private static final int HEIGHT = 13;

  /**
   * Gets height
   * 
   * @return Returns height
   */
  @Override
  public int getIconHeight() {
    return WIDTH;
  }

  /**
   * Gets width
   * 
   * @return Returns width
   */
  @Override
  public int getIconWidth() {
    return HEIGHT;
  }

  /**
   * Paints icon
   * 
   * @param c Component
   * @param g Graphics
   * @param x x coordinate
   * @param y y coordinate
   */
  @Override
  public void paintIcon(Component c, Graphics g, int x, int y) {

    g.setColor(WHITE_LINE_COLOR);
    g.drawLine(0, 12, 12, 0);
    g.drawLine(5, 12, 12, 5);
    g.drawLine(10, 12, 12, 10);

    g.setColor(GRAY_LINE_COLOR);
    g.drawLine(1, 12, 12, 1);
    g.drawLine(2, 12, 12, 2);
    g.drawLine(3, 12, 12, 3);

    g.drawLine(6, 12, 12, 6);
    g.drawLine(7, 12, 12, 7);
    g.drawLine(8, 12, 12, 8);

    g.drawLine(11, 12, 12, 11);
    g.drawLine(12, 12, 12, 12);

  }
}  // class AngledLinesWindowsCornerIcon
