/*
 * Project: Example game for creation of installer
 * Authors: Ing. Jan Kouřil
 *          Ing. Jaroslav Dytrych idytrych@fit.vutbr.cz
 * File: canvasWidget.java
 */

/**
 * @file canvasWidget.java
 *
 * @brief Canvas for picture
 */

import java.awt.Color;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import javax.swing.JComponent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import javax.imageio.ImageIO;
import javax.swing.JOptionPane;

/**
 * Canvas for picture
 *
 * @brief Canvas for picture
 */
public class canvasWidget extends JComponent implements ActionListener, MouseListener{
    /** Matrice 4x4 */
    boolean matrix4[][];
    /** Matrice 6x6 */
    boolean matrix6[][];
    /** Matrice 8x8 */
    boolean matrix8[][];
    /** List of images */
    ArrayList<BufferedImage> images;
    /** Mode (matrice dimension) */
    int mode;
    /** Index of current image in list */
    int currentIndex;
    /** Action listener */
    ActionListener listener;
    /** Actual image */
    BufferedImage actualImage;
    
    /**
     * Constructor
     */
    public canvasWidget(){
        matrix8=new boolean[8][8];
        matrix6=new boolean[6][6];
        matrix4=new boolean[4][4];
        actualImage=null;
        listener=null;
        mode=8;
        images=new ArrayList<BufferedImage>();
        this.addMouseListener(this);
    }
    
    /**
     * Sets action listener
     * 
     * @param a Action listener
     */
    public void setActionListener(ActionListener a){
        listener=a;
    }
    
    /**
     * Adds image into list
     * 
     * @param f File with image
     */
    public void addImage(File f){
        BufferedImage img = null;
        try {
            img = ImageIO.read(f);
        } catch (IOException e) {
            return;
        }
        images.add(img);
    }
    
    /**
     * Sets mode
     * 
     * @param m Mode (dimension of matrice)
     */
    public void setMode(int m){
        mode=m;
        repaint();
    }
    
    /**
     * Starts game (shows and covers first image)
     */
    private void start(){
        currentIndex=0;
        if (images.isEmpty()){
            Object[] options = { "OK" };
            JOptionPane.showOptionDialog(null, "No images inserted", "Warning",
            JOptionPane.DEFAULT_OPTION, JOptionPane.WARNING_MESSAGE,
            null, options, options[0]);
            return;
        }
        actualImage=(BufferedImage)images.get(currentIndex);
        if (listener!=null) {
            listener.actionPerformed(new ActionEvent(this,java.awt.event.ActionEvent.ACTION_PERFORMED,"Resize "+Integer.toString(actualImage.getWidth())+" "+Integer.toString(actualImage.getHeight())));
        }
        coverMatrices(); 
        if (listener!=null) {
            listener.actionPerformed(new ActionEvent(this,java.awt.event.ActionEvent.ACTION_PERFORMED,"Image 1/"+Integer.toString(images.size())));
        }
    }
    
    /**
     * Covers matrices
     */
    private void coverMatrices(){
        for (int i=0;i<8;i++) {
            for (int j=0;j<8;j++) {
                matrix8[i][j]=false;
            }
        }
        for (int i=0;i<6;i++) {
            for (int j=0;j<6;j++) {
                matrix6[i][j]=false;
            }
        }
        for (int i=0;i<4;i++) {
            for (int j=0;j<4;j++) {
                matrix4[i][j]=false;
            }
        }
        repaint();
    }
    
    /**
     * Clear (uncovers) matrices
     */
    private void clearMatrices(){
        for (int i=0;i<8;i++) {
            for (int j=0;j<8;j++) {
                matrix8[i][j]=true;
            }
        }
        for (int i=0;i<6;i++) {
            for (int j=0;j<6;j++) {
                matrix6[i][j]=true;
            }
        }
        for (int i=0;i<4;i++) {
            for (int j=0;j<4;j++) {
                matrix4[i][j]=true;
            }
        }
        repaint();
    }
    
    /**
     * Paints canvas
     */
    @Override
    public void paint(Graphics g) {
        if (actualImage!=null){
            boolean mat[][]=mode==8?matrix8:mode==6?matrix6:matrix4;
            g.drawImage(actualImage, 0, 0, this);
            int w=this.getWidth()/mode;
            int h=this.getHeight()/mode;
            for (int i=0;i<mode;i++){
                for (int j=0;j<mode;j++){
                    if (!mat[i][j]){
                        g.setColor(Color.white);
                        g.fillRect(i*w, j*h, w, h);
                        FontMetrics fm=g.getFontMetrics();
                        int letter='A'+i;
                        int number='1'+j;
                        String s=Character.toString((char)letter)+Character.toString((char)number);
                        int xOffset=(w-fm.stringWidth(s))/2;
                        int yOffset=(fm.getAscent()+(h-(fm.getAscent()+fm.getDescent()))/2);
                        g.setColor(Color.black);
                        g.drawString(s, i*w+xOffset, j*h+yOffset);
                        g.drawRect(i*w, j*h, w, h);
                    }
                }
            }
        }
    }
    
    /**
     * Switches to next image 
     */
    private void next(){
        if (images.isEmpty()) {
            return;
        }
        currentIndex=(currentIndex+1)%images.size();
        actualImage=(BufferedImage)images.get(currentIndex);
        if (listener!=null) {
            listener.actionPerformed(new ActionEvent(this,java.awt.event.ActionEvent.ACTION_PERFORMED,"Resize "+Integer.toString(actualImage.getWidth())+" "+Integer.toString(actualImage.getHeight())));
        }
        coverMatrices();
        if (listener!=null) {
            listener.actionPerformed(new ActionEvent(this,java.awt.event.ActionEvent.ACTION_PERFORMED,"Image "+Integer.toString(currentIndex)+"/"+Integer.toString(images.size())));
        }
    }
    
    /**
     * Switches to previous image 
     */
    private void prev(){
        if (images.isEmpty()) {
            return;
        }
        currentIndex--;
        if (currentIndex<0) {
            currentIndex=images.size()-1;
        }
        actualImage=(BufferedImage)images.get(currentIndex);
        if (listener!=null) {
            listener.actionPerformed(new ActionEvent(this,java.awt.event.ActionEvent.ACTION_PERFORMED,"Resize "+Integer.toString(actualImage.getWidth())+" "+Integer.toString(actualImage.getHeight())));
        }
        coverMatrices();
        if (listener!=null) {
            listener.actionPerformed(new ActionEvent(this,java.awt.event.ActionEvent.ACTION_PERFORMED,"Image "+Integer.toString(currentIndex)+"/"+Integer.toString(images.size())));
        }
    }
    

    /**
     * Handles events
     * 
     * @param ae Name of event action
     */
    @Override
    public void actionPerformed(ActionEvent ae) {
        String command=ae.getActionCommand();
        if ("Start".equals(command)) {
            start();
        } else if ("Next".equals(command)) {
            next();
        } else if ("Prev".equals(command)) {
            prev();
        } else if ("Clear".equals(command)) {
            clearMatrices();
        } else if ("Cover".equals(command)) {
            coverMatrices();
        }
    }

    /**
     * Handles mouse click event
     * 
     * @param me Mouse event
     */
    @Override
    public void mouseClicked(MouseEvent me) {
    }

    /**
     * Handles mouse press event
     * 
     * @param me Mouse event
     */
    @Override
    public void mousePressed(MouseEvent me) {
        boolean mat[][]=mode==8?matrix8:mode==6?matrix6:matrix4;
        int w=this.getWidth();
        int h=this.getHeight();
        int matX=me.getX()/(w/mode);
        int matY=me.getY()/(h/mode);
        mat[matX][matY]=true;
        repaint();
    }

    /**
     * Handles mouse release event
     * 
     * @param me Mouse event
     */
    @Override
    public void mouseReleased(MouseEvent me) {
        
    }

    /**
     * Handles mouse enter event
     * 
     * @param me Mouse event
     */
    @Override
    public void mouseEntered(MouseEvent me) {
        
    }

    /**
     * Handles mouse exit event
     * 
     * @param me Mouse event
     */
    @Override
    public void mouseExited(MouseEvent me) {
        
    }

}  // public class canvasWidget
