#!/usr/bin/env python
# -*- coding: utf-8 -*-

#
# Project: Example game for creation of installer
# Authors: Ing. Jan Kouřil
#          Ing. Jaroslav Dytrych dytrych@fit.vut.cz
#          Miroslav Karásek xkaras31@stud.fit.vutbr.cz
# File: kufr.py
#
from PySide2 import QtWidgets, QtGui, QtCore
from PySide2.QtCore import QObject, Signal, Slot
import sys
from pprint import pprint

## Canvas for imasge
class CanvasWidget(QtWidgets.QWidget ):
    actualImage = None
    
    ## Constructor
    def __init__(self, parent=None):
        super(CanvasWidget, self).__init__(parent)
        self.mode=8
        self.currentIndex=0
        self.setAutoFillBackground(True)
        self.list = []
        self.parent = parent
        self.coverMatrices()
    
    ## Adds image
    def addImage(self, string):
        if string:
            self.list.append(QtGui.QImage(string))
    
    ## Sets mode (dimension of matrice)
    def setMode(self, m):
        self.mode=m
        self.repaint()
    
    ## Starts game (shows and covers first image)
    def start(self):
        self.currentIndex=0;
        if len(self.list)==0:
            QtWidgets.QMessageBox.warning(None,"Warning","No images inserted")
            return
        self.actualImage=self.list[0]        
        self.resize(self.actualImage.width(),self.actualImage.height())
        self.parent.resize(self.actualImage.width(),self.actualImage.height())
        self.coverMatrices();
        self.setStatusTip("Image 1/"+str(len(self.list)));
    
    ## Switches to next image
    def __next__(self):
        if len(self.list)==0:
            QtWidgets.QMessageBox.warning(None,"Warning","No images inserted")
            return
        self.currentIndex=(self.currentIndex+1)%len(self.list);
        self.actualImage=self.list[self.currentIndex]  
        self.resize(self.actualImage.width(),self.actualImage.height())
        self.parent.resize(self.actualImage.width(),self.actualImage.height())
        self.coverMatrices();
        self.setStatusTip("Image "+str(self.currentIndex+1)+"/"+str(len(self.list)));
        
    ## Switches to previous image 
    def prev(self):
        if len(self.list)==0:
            QtWidgets.QMessageBox.warning(None,"Warning","No images inserted")
            return
        self.currentIndex-=1
        if (self.currentIndex<0):
            self.currentIndex=len(self.list)-1;
        self.actualImage=self.list[self.currentIndex]    
        self.resize(self.actualImage.width(),self.actualImage.height())
        self.parent.resize(self.actualImage.width(),self.actualImage.height())
        self.coverMatrices();
        self.setStatusTip("Image "+str(self.currentIndex+1)+"/"+str(len(self.list)));
    
    ## Covers matrices
    def coverMatrices(self):
        self.matrix = [
            [False,False,False,False,False,False,False,False],
            [False,False,False,False,False,False,False,False],
            [False,False,False,False,False,False,False,False],
            [False,False,False,False,False,False,False,False],
            [False,False,False,False,False,False,False,False],
            [False,False,False,False,False,False,False,False],
            [False,False,False,False,False,False,False,False],
            [False,False,False,False,False,False,False,False]
        ]
        self.repaint()
    
    ## Clear (uncovers) matrices
    def clearMatrices(self):
        self.matrix = [
            [True,True,True,True,True,True,True,True],
            [True,True,True,True,True,True,True,True],
            [True,True,True,True,True,True,True,True],
            [True,True,True,True,True,True,True,True],
            [True,True,True,True,True,True,True,True],
            [True,True,True,True,True,True,True,True],
            [True,True,True,True,True,True,True,True],
            [True,True,True,True,True,True,True,True]
        ]
        self.repaint()
    
    ## Paints canvas
    def paintEvent(self, e):
        if (self.actualImage):
            painter = QtGui.QPainter(self);
            painter.drawImage(QtCore.QPoint(0,0),self.actualImage);
            w=self.width()/self.mode;
            h=self.height()/self.mode;
            for i in range(0,self.mode):
                for j in range(0,self.mode):
                    if (not self.matrix[i][j]):
                        rect=QtCore.QRect(i*w,j*h,w,h);
                        painter.setPen(QtGui.QColor("black"));
                        painter.fillRect(rect,QtCore.Qt.white);
                        a = ['A','B','C','D','E','F','G','G',]
                        painter.drawText(rect,QtCore.Qt.AlignCenter,a[i]+str(j+1));
                        painter.drawRect(rect);
            painter.end()
    
    ## Handles mouse press event
    def mousePressEvent(self, e):
        w=self.width()
        h=self.height()
        matX=int(e.x()/(w/self.mode))
        matY=int(e.y()/(h/self.mode))
        self.matrix[matX][matY]=True
        self.repaint()

## Program main window
class MainWin(QtWidgets.QMainWindow ):
  
    ## Constructor
    def __init__(self, parent=None):
        super(MainWin, self).__init__(parent)          # volani rodice
        self.setWindowTitle("Moje první aplikace") # zmena nazvu okna
        # menu bar
        menubar= QtWidgets.QMenuBar(self);

        # File menu
        file=QtWidgets.QMenu("File",menubar)
        addFile=QtWidgets.QAction("Add file",file)
        addFile.setShortcut(QtGui.QKeySequence("Ctrl+F"))
        file.addAction(addFile)
        addDirectory=QtWidgets.QAction("Add directory",file)
        addDirectory.setShortcut(QtGui.QKeySequence("Ctrl+D"))
        file.addAction(addDirectory)
        file.addSeparator()
        quit=QtWidgets.QAction("Quit",file)
        quit.setShortcut(QtGui.QKeySequence("Ctrl+Q"))
        file.addAction(quit)
        menubar.addMenu(file)
        # Game menu
        game=QtWidgets.QMenu("Game",menubar);
        self.start=QtWidgets.QAction("Start",game);
        self.start.setShortcut(QtGui.QKeySequence("Ctrl+S"));
        game.addAction(self.start);
        game.addSeparator();
        self.next=QtWidgets.QAction("Next",game);
        self.next.setShortcut(QtGui.QKeySequence(QtCore.Qt.Key_Right));
        game.addAction(self.next);
        self.previous=QtWidgets.QAction("Previous",game);
        self.previous.setShortcut(QtGui.QKeySequence(QtCore.Qt.Key_Left));
        game.addAction(self.previous);
        game.addSeparator();
        self.clear=QtWidgets.QAction("Clear",game);
        self.clear.setShortcut(QtGui.QKeySequence(QtCore.Qt.Key_Return));
        game.addAction(self.clear);
        self.cover=QtWidgets.QAction("Cover",game);
        self.cover.setShortcut(QtGui.QKeySequence(QtCore.Qt.Key_Escape));
        game.addAction(self.cover);
        menubar.addMenu(game);
        # Grid menu
        grid=QtWidgets.QMenu("Grid",menubar);
        self.four=QtWidgets.QAction("4x4",grid);
        self.four.setShortcut(QtGui.QKeySequence(QtCore.Qt.Key_4));
        self.four.setCheckable(True);
        grid.addAction(self.four);
        self.six=QtWidgets.QAction("6x6",grid);
        self.six.setShortcut(QtGui.QKeySequence(QtCore.Qt.Key_6));
        self.six.setCheckable(True);
        grid.addAction(self.six);
        self.eight=QtWidgets.QAction("8x8",grid);
        self.eight.setShortcut(QtGui.QKeySequence(QtCore.Qt.Key_8));
        self.eight.setCheckable(True);
        self.eight.setChecked(True);
        grid.addAction(self.eight);
        menubar.addMenu(grid);
        # Attach menu bar to self window
        self.setMenuBar(menubar);
        # Create canvas for picture
        self.canvas=CanvasWidget(self);
        self.setCentralWidget(self.canvas);
        # Create status bar
        status=QtWidgets.QStatusBar(self);
        self.setStatusBar(status);
        # Set window title and initial size
        self.setWindowTitle("Kufr");
        self.resize(800,600);
        # Set initial status
        statusLabel=QtWidgets.QLabel("Ready.",self);
        status.addWidget(statusLabel,0);

        # Connect signals
        addFile.triggered.connect(self.aFile)
        addDirectory.triggered.connect(self.aDirectory)
        quit.triggered.connect(QtCore.QCoreApplication.instance().quit)
        
        self.start.triggered.connect(self.canvas.start)
        self.next.triggered.connect(self.canvas.__next__)
        self.previous.triggered.connect(self.canvas.prev)
        self.clear.triggered.connect(self.canvas.clearMatrices)
        self.cover.triggered.connect(self.canvas.coverMatrices)
        
        self.four.triggered.connect(self.set4)
        self.six.triggered.connect(self.set6)
        self.eight.triggered.connect(self.set8)

    ## Handles resize event
    def resizeCanvas(self,resizeEvent):
        self.resizeCanvas(resizeEvent.size, resizeEvent.oldSize)

    ## Adds file with image
    def aFile(self):
        string=QtWidgets.QFileDialog.getOpenFileName(None,"Open file","","Images (*.gif *.jpeg *.pbm *.bmp *.pgm *.ppm *.tiff *.xbm *.png *.xpm *.jpg)");
        if (string):
            self.canvas.addImage(string[0])

    ## Adds directory with images
    def aDirectory(self):
        string=QtWidgets.QFileDialog.getExistingDirectory(None,"Open directory")
        if (string):
            dir = QtCore.QDir(path=string)
            entries=dir.entryList(["*.gif","*.jpeg","*.pbm","*.bmp","*.pgm","*.ppm","*.tiff","*.xbm","*.png","*.xpm","*.jpg"]);
            for file in entries:
                self.canvas.addImage(string+"/"+file);

    ##  Handler for Grid 4x4 menu action
    def set4(self):
        self.four.setChecked(True)
        self.six.setChecked(False)
        self.eight.setChecked(False)
        self.canvas.setMode(4)

    ##  Handler for Grid 6x6 menu action
    def set6(self):
        self.four.setChecked(False)
        self.six.setChecked(True)
        self.eight.setChecked(False)
        self.canvas.setMode(6)

    ##  Handler for Grid 8x8 menu action
    def set8(self):
        self.four.setChecked(False)
        self.six.setChecked(False)
        self.eight.setChecked(False)
        self.canvas.setMode(8);

if __name__ == "__main__":
    appl = QtWidgets.QApplication(sys.argv)  # create PySide application
    myWin = MainWin()  # create main window
    myWin.show()  # show main window
    appl.exec_()  # start application

