/*
 * Project: Example game for creation of installer
 * Authors: Ing. Jan Kouřil
 *          Ing. Jaroslav Dytrych idytrych@fit.vutbr.cz
 * File: canvasWidget.cpp
 */

/**
 * @file canvasWidget.cpp
 *
 * @brief Canvas for picture
 */

#include "canvasWidget.h"
#include "QPainter"
#include "QMouseEvent"
#include "QMessageBox"
#include "QMainWindow"
#include "QStatusBar"

/**
 * Constructor
 */
CanvasWidget::CanvasWidget(QWidget *parent) :
    QWidget(parent)
{
    mode=8;
    currentIndex=0;
    setAutoFillBackground(true);
    matrix8=new bool*[8];
    matrix6=new bool*[6];
    matrix4=new bool*[4];
    for (int i=0;i<8;i++)
        matrix8[i]=new bool[8];
    for (int i=0;i<6;i++)
        matrix6[i]=new bool[6];
    for (int i=0;i<4;i++)
        matrix4[i]=new bool[4];
}

/**
 * Destructor
 */
CanvasWidget::~CanvasWidget(){
    for (int i=0;i<8;i++)
        delete [] matrix8[i];
    delete [] matrix8;
    for (int i=0;i<6;i++)
        delete [] matrix6[i];
    delete [] matrix6;
    for (int i=0;i<4;i++)
        delete [] matrix4[i];
    delete [] matrix4;
}

/** 
 * Adds image 
 */
void CanvasWidget::addImage(QString str){
    list.append(QImage(str));
}

/** 
 * Sets mode 
 * 
 * @param m Mode (dimension of matrice)
 */
void CanvasWidget::setMode(int m){
    mode=m;
    repaint();
}

/**
 * Starts game (shows and covers first image)
 */
void CanvasWidget::start(){
    currentIndex=0;
    if (list.size()==0){
        QMessageBox::warning(NULL,"Warning","No images inserted");
        return;
    }
    actualImage=list[0];
    emit(resizeRequest(actualImage.width(),actualImage.height()));
    coverMatrices();
    emit(statusRequest("Image 1/"+QString::number(list.size())));
}

/**
 * Covers matrices
 */
void CanvasWidget::coverMatrices(){
    for (int i=0;i<8;i++)
        for (int j=0;j<8;j++)
            matrix8[i][j]=false;
    for (int i=0;i<6;i++)
        for (int j=0;j<6;j++)
            matrix6[i][j]=false;
    for (int i=0;i<4;i++)
        for (int j=0;j<4;j++)
            matrix4[i][j]=false;
    repaint();
}

/**
 * Clear (uncovers) matrices
 */
void CanvasWidget::clearMatrices(){
    for (int i=0;i<8;i++)
        for (int j=0;j<8;j++)
            matrix8[i][j]=true;
    for (int i=0;i<6;i++)
        for (int j=0;j<6;j++)
            matrix6[i][j]=true;
    for (int i=0;i<4;i++)
        for (int j=0;j<4;j++)
            matrix4[i][j]=true;
    repaint();
}

/**
 * Paints canvas
 */
void CanvasWidget::paintEvent(QPaintEvent *){
    if (!actualImage.isNull()){
        QPainter painter(this);
        painter.drawImage(QPoint(0,0),actualImage);
        bool **mat=mode==8?matrix8:mode==6?matrix6:matrix4;  // select matrice
        int w=width()/mode;
        int h=height()/mode;
        for (int i=0;i<mode;i++){
            for (int j=0;j<mode;j++){
                if (!mat[i][j]){
                    QRect rect(i*w,j*h,w,h);
                    painter.setPen(QColor("black"));
                    painter.fillRect(rect,Qt::white);
                    painter.drawText(rect,Qt::AlignCenter,QString('A'+i)+QString(j+'1'));
                    painter.drawRect(rect);
                }
            }
        }
    }
}

/**
 * Handles mouse press event
 * 
 * @param e Mouse event
 */
void CanvasWidget::mousePressEvent(QMouseEvent *e){
    int w=width();
    int h=height();
    int matX=e->x()/(w/mode);
    int matY=e->y()/(h/mode);
    bool **mat=mode==8?matrix8:mode==6?matrix6:matrix4;  // select matrice
    mat[matX][matY]=true;
    repaint();
}

/** 
 * Switches to next image 
 */
void CanvasWidget::next(){
    if (list.size()==0)
        return;
    currentIndex=(currentIndex+1)%list.size();
    actualImage=list[currentIndex];
    emit(resizeRequest(actualImage.width(),actualImage.height()));
    coverMatrices();
    emit(statusRequest("Image "+QString::number(currentIndex+1)+"/"+QString::number(list.size())));
}

/** 
 * Switches to previous image 
 */
void CanvasWidget::prev(){
    if (list.size()==0)
        return;
    currentIndex--;
    if (currentIndex<0){
        currentIndex=list.size()-1;
    }
    actualImage=list[currentIndex];
    emit(resizeRequest(actualImage.width(),actualImage.height()));
    coverMatrices();
    emit(statusRequest("Image "+QString::number(currentIndex+1)+"/"+QString::number(list.size())));
}
